"use client";

import { Suspense, useEffect, useMemo, useState } from "react";
import { useRouter } from "next/navigation";

import dynamic from "next/dynamic";
import Swal from "sweetalert2";
import { toast } from "react-toastify";
// import CustomDataTable from "../../dataTable/customDataTable";
import { getBookColumns } from "./columns";
import { updateBookAction } from "@/lib/actions/bookActions";
import Select from "react-select";
import selectStyles from "@/app/components/selectStyles";
import BooksListShimmer from "@/app/components/BookListShimmer";

// ✅ Lazy load CustomDataTable
const CustomDataTable = dynamic(() => import("../../../(views)/dataTable/customDataTable"), {
  loading: () => <BooksListShimmer rows={6} columns={5} />, // shimmer placeholder while table loads
  ssr: false,
});


export default function BooksList({ initialBooks,initialBookYears, initialBookEntries, initialBookId, role }) {
  const [data, setData] = useState(initialBooks || []);
  const [loading, setLoading] = useState(true);
  const [filteredData, setFilteredData] = useState([]);
  const [yearOptions, setYearOptions] = useState([]);
  const [selectedYear, setSelectedYear] = useState(null);
  const router = useRouter();

   /* useEffect(() => {
    const currentYear = new Date().getFullYear();
    const years = [];
    for (let i = 0; i < 10; i++) {
      const year = currentYear + i;
      years.push({ value: year.toString(), label: `${year}` });
    }
    setYearOptions(years);
    setSelectedYear({ value: currentYear.toString(), label: `${currentYear}` });
  }, []); */
  useEffect(() => {
    if (initialBookYears && initialBookYears.length > 0) {
      const formattedYears = initialBookYears.map((y) => ({
        value: y.book_year.toString(),
        label: y.book_year.toString(),
      }));
      setYearOptions(formattedYears);
      //setSelectedYear(formattedYears[0]); // ✅ sets first year as default
      //setSelectedYear({ value: "all", label: "All Years" });
    }
  }, [initialBookYears]);

  useEffect(() => {
    setData(initialBooks || []);
    setFilteredData(initialBooks || []);
    setLoading(false);
  }, [initialBooks]);

  
  
  //Filter Book
  useEffect(() => {
    const filtered = data.filter((item) => {
        //const itemDate = new Date(item.book_createdAt).getFullYear().toString();
        //const matchesYear = selectedYear ? itemDate === selectedYear.value : true;
          const matchesYear = selectedYear ? item.book_year === selectedYear.value : true;
        return (
          matchesYear 
        );
      });
    setFilteredData(filtered);
  }, [selectedYear, data]);

  const handleDelete = async (bookId, bookName) => {
    const result = await Swal.fire({
      title: "Are you sure?",
      text: `Do you really want to delete the book named "${bookName}"?`,
      icon: "warning",
      showCancelButton: true,
      confirmButtonColor: "#d33",
      cancelButtonColor: "#3085d6",
      confirmButtonText: "Yes, delete it!",
    });

    if (!result.isConfirmed) return;

    try {
      const payload = {
        book_isdelete: "inactive",
      };
      console.log(bookId);
      const res = await updateBookAction(bookId, payload);
      if (res?.success) {
        // ✅ Filter out deleted book from local state
         setData((prevData) =>
        prevData.filter((data) => data.book_pid !== bookId)
      );
        toast.success("Book has been deleted.");
      } else {
        toast.error(res?.message || "Failed to delete book.");
      }

    } catch (error) {
      toast.error(error.message || "Error deleting book");
    }
  };

  const combinedBook = initialBookId.map((id, index) => ({
    id,
    value: initialBookEntries[index],
  }));

  const columns = useMemo(() => getBookColumns(router, handleDelete, combinedBook, role), [router]);

  return (
    <div className="max-w-full px-2 py-2 mx-auto rounded-lg  space-y-4">
      <header>
        <div className="flex justify-between items-center border-b border-purple-500">
          <div className="mb-1">
            <h2 className="text-xl font-semibold text-gray-800">Book Lists</h2>
            <p className="text-gray-500 mt-1 text-sm">
              Below is a list of all books.
            </p>
          </div>
          {role !== 3 && (
          <button
            onClick={() => router.push("/books/create")}
            className="bg-gradient-to-r from-[#8C3FE1] to-[#E557AE] hover:from-[#6B3DC6] hover:to-[#C64A9B] text-white px-5 py-2 cursor-pointer rounded-md shadow-md text-sm font-medium transition-transform transform hover:scale-105"
          >
            + New Book
          </button>
          )}
        </div>
      </header>
      
            {loading ? (
              <BooksListShimmer rows={6} columns={5} />
            ) : (
              <Suspense fallback={<BooksListShimmer rows={6} columns={5} />}>
<div className="bg-purple-50 border border-purple-200 rounded-lg p-2 shadow-sm">
        <div className="flex flex-wrap gap-4 items-center justify-start">
          {/* Year Filter */}
          <div className="flex flex-col">
            <label className="text-xs text-gray-600 font-semibold mb-1">
              Year
            </label>
            <Select
              instanceId="year-filter"
              options={yearOptions}
              value={selectedYear}
              onChange={setSelectedYear}
              placeholder="Select Year"
              isClearable
              className="w-33 text-sm"
              menuPortalTarget={document.body}
              styles={selectStyles}
            />
          </div>
          
          {/* Clear Filters Button */}
          <div className="pt-5">
            <button
              onClick={() => {
                setSelectedYear(null);
              }}
              className="text-sm px-4 py-2 bg-purple-100 hover:bg-purple-200 text-purple-700 rounded transition-all duration-200 cursor-pointer"
            >
              Clear Filters
            </button>
          </div>
        </div>
      </div>
      <CustomDataTable
        columns={columns}
        data={filteredData}
        progressPending={loading}
        defaultSort={{ key: "Working Title", direction: "ascending" }}
      />
      </Suspense>
      )}
    </div>
  );
}