"use client";

import React, { useEffect, useState } from "react";
import { Controller, useForm ,useWatch} from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { bookSchema } from "@/validation/validation";
import { toast } from "react-toastify";
import { useRouter } from "next/navigation";
import { FiArrowRightCircle, FiLoader } from "react-icons/fi";
import { createBookAction, updateBookAction } from "@/lib/actions/bookActions";
import BackButton from "@/app/components/BackButton";
import Select from "react-select";
export default function CreateBookForm({ book, bookId, user_pid,project_user,altproject_user }) {
  const [loading, setLoading] = useState(false);
  const isEditMode = !!bookId;
  const router = useRouter();
  const {
    register,
    handleSubmit,
    control,
    formState: { errors },
    reset,
    setValue,
  } = useForm({
    resolver: zodResolver(bookSchema),
    mode: "onBlur",
    defaultValues: {
      book_year: "",
      book_title: "",
      book_lead_editor_name: "",
      book_id: "",
      book_project_manager: "",
      book_online_isbn: "",
      book_author_invitations_sent: "",
      book_au_invited: "",
      book_au_accepted: "",
      book_chap_due_date: "",
      book_drop_dead_date: "",
      book_ch_sumitted: "",
      book_ch_accepted: "",
      book_last_due_production: "",
      
      book_alter_project_manager: "",
      book_note_comments: "",
    },
  });
  const formatDate = (dateString) => {
    if (!dateString) return "";
    return new Date(dateString).toISOString().split("T")[0];
  };
  // Hydrate form with book data if editing
  useEffect(() => {
    if (book) {
      // console.log(book,"book_alter_project_manager");
      setValue("book_year", book.book_year);
      setValue("book_title", book.book_title);
      setValue("book_lead_editor_name", book.book_LeadEditorName);
      setValue("book_id", book.book_CustomId);
      setValue("book_online_isbn", book.book_OnlineIsbn);
      setValue("book_author_invitations_sent", formatDate(book.book_AuthorInvitationsSent));
      setValue("book_au_invited", book.book_AuInvited);
      setValue("book_au_accepted", book.book_AuAccepted);
      setValue("book_chap_due_date", formatDate(book.book_ChapDueDate));
      setValue("book_drop_dead_date", formatDate(book.book_DropDeadDate));
      setValue("book_ch_sumitted", book.book_ChSubmitted);
      setValue("book_ch_accepted", book.book_ChAccepted);
      setValue("book_last_due_production", formatDate(book.book_LastDueProduction));
      setValue("book_project_manager", String(book.book_ProjectManager));
      setValue("book_alter_project_manager", book.book_alter_project_manager);
      setValue("book_note_comments", book.book_NoteComments);
    } else {
      reset();
    }
  }, [book, setValue, reset]);
  const onSubmit = async (data) => {
    setLoading(true);
    try {
      const payload = {
        book_year: data.book_year,
        book_title: data.book_title,
        book_LeadEditorName: data.book_lead_editor_name,
        book_CustomId: data.book_id,
        book_OnlineIsbn: data.book_online_isbn,
        book_AuthorInvitationsSent: data.book_author_invitations_sent,
        book_AuInvited: Number(data.book_au_invited),
        book_AuAccepted: Number(data.book_au_accepted),
        book_ChapDueDate: data.book_chap_due_date,
        book_DropDeadDate: data.book_drop_dead_date,
        book_ChSubmitted: Number(data.book_ch_sumitted),
        book_ChAccepted: Number(data.book_ch_accepted),
        book_LastDueProduction: data.book_last_due_production,
        book_ProjectManager: Number(data.book_project_manager),
        book_AltProjectManager: data.book_alter_project_manager ? Number(data.book_alter_project_manager) : null,
        book_NoteComments: data.book_note_comments,
        // book_createdBy: user_pid,
      };
      
      if (isEditMode) {
        payload.book_updatedBy = user_pid;
      } else {
        payload.book_createdBy = user_pid;
      }
      if (isEditMode) {
        console.log(data, "data in update");
        const res = await updateBookAction(bookId, payload);
        if (res?.success) {
          router.push("/books/list");
          toast.success("Book updated successfully!");
        } else {
          toast.error(res?.message || "Failed to update book.");
        }
      } else {
        const res = await createBookAction(payload);
        if (res?.success) {
          router.push("/books/list");
          toast.success("Book created successfully!");
        } else {
          toast.error(res?.message || "Failed to create book.");
        }
      }
    } catch (error) {
      toast.error(error.message || "Unexpected error occurred");
      console.error(error);
    } finally {
      setLoading(false);
    }
  };
  const handleReset = () => {
    reset();
    toast.info("Form reset.");
  };

  const selectUserList = project_user.map((s) => ({
    value: String(s.user_pid),
    label: s.user_name.charAt(0).toUpperCase() + s.user_name.slice(1),
  }));

  const selectedProjectManager = useWatch({
    control,
    name: "book_project_manager",
  });

  /*  const selectAltUserList = project_user.map((s) => ({
    value: String(s.user_pid),
    label: s.user_name.charAt(0).toUpperCase() + s.user_name.slice(1),
  }));  */

  // Filter alternative manager list dynamically
  const selectAltUserList = project_user
    .filter((s) => String(s.user_pid) !== String(selectedProjectManager))
    .map((s) => ({
      value: String(s.user_pid),
      label: s.user_name.charAt(0).toUpperCase() + s.user_name.slice(1),
    }));

  return (
    <div className="max-w-full px-4 py-4 mx-auto e rounded-lg  space-y-4 ">
      <header>
        <div className="flex justify-between items-center border-b border-purple-500">
          <div className="mb-1">
            <h2 className="text-xl font-semibold text-gray-800">
              {isEditMode ? "Edit Book" : "Create New Book"}
            </h2>
            <p className="text-gray-500 mt-1 text-sm">
              {isEditMode
                ? "Modify the details of the selected book."
                : "Fill in the details below to add a new book."}
            </p>
          </div>
          <BackButton />
        </div>
      </header>
      <form
        onSubmit={handleSubmit(onSubmit)}
        className="grid grid-cols-1 md:grid-cols-2 gap-8 max-w-4xl mx-auto"
      >
        {/* Left Column */}
        <div className="space-y-4">
          {/* Year of Working Title */}
          <div>
            <label
              htmlFor="book_year"
              className="block text-sm font-medium text-gray-700 mb-1"
            >
              Year of Working Title <span className="text-red-500">*</span>
            </label>
            <input
              id="book_year"
              type="text"
              placeholder="Enter year of working title"
              {...register("book_year")}
              className={`w-full px-4 py-1.5 rounded-md border placeholder:text-sm ${errors.book_year ? "border-red-500" : "border-gray-300"
                } focus:outline-none focus:ring-2 focus:ring-blue-500`}
            />
            {errors.book_year && (
              <p className="text-sm text-red-500 mt-1">
                {errors.book_year.message}
              </p>
            )}
          </div>
          {/* Book Title */}
          <div>
            <label
              htmlFor="book_title"
              className="block text-sm font-medium text-gray-700 mb-1"
            >
              Book Title <span className="text-red-500">*</span>
            </label>
            <input
              id="book_title"
              type="text"
              placeholder="Enter book title"
              {...register("book_title")}
              className={`w-full px-4 py-1.5 rounded-md border placeholder:text-sm ${errors.book_title ? "border-red-500" : "border-gray-300"
                } focus:outline-none focus:ring-2 focus:ring-blue-500`}
            />
            {errors.book_title && (
              <p className="text-sm text-red-500 mt-1">
                {errors.book_title.message}
              </p>
            )}
          </div>
          {/* Lead Editor Name */}
          <div>
            <label
              htmlFor="book_lead_editor_name"
              className="block text-sm font-medium text-gray-700 mb-1"
            >
              Lead Editor Name <span className="text-red-500">*</span>
            </label>
            <input
              id="book_lead_editor_name"
              type="text"
              placeholder="Enter lead editor name"
              {...register("book_lead_editor_name")}
              className={`w-full px-4 py-1.5 rounded-md border placeholder:text-sm ${errors.book_lead_editor_name
                ? "border-red-500"
                : "border-gray-300"
                } focus:outline-none focus:ring-2 focus:ring-blue-500`}
            />
            {errors.book_lead_editor_name && (
              <p className="text-sm text-red-500 mt-1">
                {errors.book_lead_editor_name.message}
              </p>
            )}
          </div>
          {/* Book ID */}
          <div>
            <label
              htmlFor="book_id"
              className="block text-sm font-medium text-gray-700 mb-1"
            >
              Book ID <span className="text-red-500">*</span>
            </label>
            <input
              id="book_id"
              type="text"
              placeholder="Enter Book ID"
              {...register("book_id")}
              className={`w-full px-4 py-1.5 rounded-md border placeholder:text-sm ${errors.book_id ? "border-red-500" : "border-gray-300"
                } focus:outline-none focus:ring-2 focus:ring-blue-500`}
            />
            {errors.book_id && (
              <p className="text-sm text-red-500 mt-1">
                {errors.book_id.message}
              </p>
            )}
          </div>
          
          {/* Online ISBN */}
          <div>
            <label
              htmlFor="book_online_isbn"
              className="block text-sm font-medium text-gray-700 mb-1"
            >
              Online ISBN <span className="text-red-500"></span>
            </label>
            <input
              id="book_online_isbn"
              type="text"
              placeholder="Enter Online ISBN"
              {...register("book_online_isbn")}
              className={`w-full px-4 py-1.5 rounded-md border placeholder:text-sm ${errors.book_online_isbn ? "border-red-500" : "border-gray-300"
                } focus:outline-none focus:ring-2 focus:ring-blue-500`}
            />
            {errors.book_online_isbn && (
              <p className="text-sm text-red-500 mt-1">
                {errors.book_online_isbn.message}
              </p>
            )}
          </div>
          {/* Author Invitations Sent (DT) */}
          <div>
            <label
              htmlFor="book_author_invitations_sent"
              className="block text-sm font-medium text-gray-700 mb-1"
            >
              Author Invitations Sent (DATE){" "}
              <span className="text-red-500"></span>
            </label>
            <input
              id="book_author_invitations_sent"
              type="date"
              placeholder="Enter author invitations sent"
              {...register("book_author_invitations_sent")}
              className={`w-full px-4 py-1.5 rounded-md border placeholder:text-sm ${errors.book_author_invitations_sent
                ? "border-red-500"
                : "border-gray-300"
                } focus:outline-none focus:ring-2 focus:ring-blue-500`}
            />
            {errors.book_author_invitations_sent && (
              <p className="text-sm text-red-500 mt-1">
                {errors.book_author_invitations_sent.message}
              </p>
            )}
          </div>
          {/* AU Invited */}
          <div>
            <label
              htmlFor="book_au_invited"
              className="block text-sm font-medium text-gray-700 mb-1"
            >
              AU Invited
            </label>
            <input
              id="book_au_invited"
              type="number"
              min="0"
              placeholder="Enter AU Invited"
              {...register("book_au_invited")}
              className={`w-full px-4 py-1.5 rounded-md border placeholder:text-sm ${errors.book_au_invited ? "border-red-500" : "border-gray-300"
                } focus:outline-none focus:ring-2 focus:ring-blue-500`}
            />
            {errors.book_au_invited && (
              <p className="text-sm text-red-500 mt-1">
                {errors.book_au_invited.message}
              </p>
            )}
          </div>
          {/* AU Accepted */}
          <div>
            <label
              htmlFor="book_au_accepted"
              className="block text-sm font-medium text-gray-700 mb-1"
            >
              AU Accepted
            </label>
            <input
              id="book_au_accepted"
              type="number"
              min="0"
              placeholder="Enter AU accepted"
              {...register("book_au_accepted")}
              className={`w-full px-4 py-1.5 rounded-md border placeholder:text-sm ${errors.book_au_accepted ? "border-red-500" : "border-gray-300"
                } focus:outline-none focus:ring-2 focus:ring-blue-500`}
            />
            {errors.book_au_accepted && (
              <p className="text-sm text-red-500 mt-1">
                {errors.book_au_accepted.message}
              </p>
            )}
          </div>
        </div>
        {/* Right Column */}
        <div className="space-y-4">
          {/* Chapter Due Date */}
          <div>
            <label
              htmlFor="book_chap_due_date"
              className="block text-sm font-medium text-gray-700 mb-1"
            >
              Drop Dead Date <span className="text-red-500"></span>
            </label>
            <input
              id="book_chap_due_date"
              type="date"
              placeholder="Enter chapter due date"
              min={new Date().toISOString().split("T")[0]}
              {...register("book_chap_due_date")}
              className={`w-full px-4 py-1.5 rounded-md border placeholder:text-sm ${errors.book_chap_due_date
                ? "border-red-500"
                : "border-gray-300"
                } focus:outline-none focus:ring-2 focus:ring-blue-500`}
            />
            {errors.book_chap_due_date && (
              <p className="text-sm text-red-500 mt-1">
                {errors.book_chap_due_date.message}
              </p>
            )}
          </div>

          {/* Ch Submitted */}
          <div>
            <label
              htmlFor="book_ch_sumitted"
              className="block text-sm font-medium text-gray-700 mb-1"
            >
              Ch Submitted
            </label>
            <input
              id="book_ch_sumitted"
              type="number"
              min="0"
              placeholder="Enter Ch Submitted"
              {...register("book_ch_sumitted")}
              className={`w-full px-4 py-1.5 rounded-md border placeholder:text-sm ${errors.book_ch_sumitted ? "border-red-500" : "border-gray-300"
                } focus:outline-none focus:ring-2 focus:ring-blue-500`}
            />
            {errors.book_ch_sumitted && (
              <p className="text-sm text-red-500 mt-1">
                {errors.book_ch_sumitted.message}
              </p>
            )}
          </div>
          {/* Ch Accepted */}
          <div>
            <label
              htmlFor="book_ch_accepted"
              className="block text-sm font-medium text-gray-700 mb-1"
            >
              Ch Accepted
            </label>
            <input
              id="book_ch_accepted"
              type="number"
              min="0"
              placeholder="Enter Ch accepted"
              {...register("book_ch_accepted")}
              className={`w-full px-4 py-1.5 rounded-md border placeholder:text-sm ${errors.book_ch_accepted ? "border-red-500" : "border-gray-300"
                } focus:outline-none focus:ring-2 focus:ring-blue-500`}
            />
            {errors.book_ch_accepted && (
              <p className="text-sm text-red-500 mt-1">
                {errors.book_ch_accepted.message}
              </p>
            )}
          </div>
          {/* Last ch due to production */}
          <div>
            <label
              htmlFor="book_last_due_production"
              className="block text-sm font-medium text-gray-700 mb-1"
            >
              Last chapter due to production{" "}
              <span className="text-red-500"></span>
            </label>
            <input
              id="book_last_due_production"
              type="date"
              placeholder="Enter last ch due to production"
              min={new Date().toISOString().split("T")[0]}
              {...register("book_last_due_production")}
              className={`w-full px-4 py-1.5 rounded-md border placeholder:text-sm ${errors.book_last_due_production
                ? "border-red-500"
                : "border-gray-300"
                } focus:outline-none focus:ring-2 focus:ring-blue-500`}
            />
            {errors.book_last_due_production && (
              <p className="text-sm text-red-500 mt-1">
                {errors.book_last_due_production.message}
              </p>
            )}
          </div>

          {/* Project Manager */}
          <div>
            <label
              htmlFor="book_project_manager"
              className="block text-sm font-medium text-gray-700 mb-1"
            >
              Project Manager <span className="text-red-500"> * </span>
            </label>
            
            <Controller
              id="book_project_manager"
              {...register("book_project_manager")} 
              control={control}
              className={`w-full px-4 py-1.5 rounded-md border placeholder:text-sm ${errors.book_project_manager ? "border-red-500" : "border-gray-300"
                } focus:outline-none focus:ring-2 focus:ring-blue-500`}
              render={({ field }) => (
                <Select
                  {...field}
                  instanceId="entry-book-pid-select"
                  options={selectUserList}
                  onChange={(selected) =>
                    field.onChange(selected ? selected.value : "")
                  }
                  value={
                    selectUserList.find(
                      (option) => option.value === field.value
                    ) || null
                  }
                  placeholder="Select an Project Manager"
                  isClearable
                  styles={{
                    control: (base) => ({
                      ...base,
                      borderColor: errors.book_project_manager
                        ? "red"
                        : base.borderColor,
                    }),
                  }}
                />
              )}
            />
            {errors.book_project_manager && (
              <p className="text-sm text-red-500 mt-1">
                {errors.book_project_manager.message}
              </p>
            )}
            
          </div>
          
          {/* Alternative Project Manager */}
          <div>
            <label
              htmlFor="book_alter_project_manager"
              className="block text-sm font-medium text-gray-700 mb-1"
            >
              Alternative Project Manager <span className="text-red-500"></span>
            </label>
            <Controller
              id="book_alter_project_manager"
              {...register("book_alter_project_manager")} 
              control={control}
              className={`w-full px-4 py-1.5 rounded-md border placeholder:text-sm ${errors.book_alter_project_manager
                ? "border-red-500"
                : "border-gray-300"
                } focus:outline-none focus:ring-2 focus:ring-blue-500`}
              render={({ field }) => (
                <Select
                  {...field}
                  instanceId="entry-book-pid-select"
                  options={selectAltUserList}
                  onChange={(selected) =>
                    field.onChange(selected ? selected.value : "")
                  }
                  value={
                    selectAltUserList.find(
                      (option) => option.value === field.value
                    ) || null
                  }
                  placeholder="Select an Alternative PM"
                  isClearable
                  styles={{
                    control: (base) => ({
                      ...base,
                      borderColor: errors.book_alter_project_manager
                        ? "red"
                        : base.borderColor,
                    }),
                  }}
                />
              )}
            />
          </div>
          <div>
            <label
              htmlFor="book_au_invited"
              className="block text-sm font-medium text-gray-700 mb-1"
            >
              Notes/Comments <span className="text-red-500"></span>
            </label>
            <textarea
              id="book_note_comments"
              placeholder="Enter comments"
              rows={4} // you can change the number of rows as needed
              {...register("book_note_comments")}
              className={`w-full px-4 py-1.5 rounded-md border placeholder:text-sm ${errors.book_note_comments
                ? "border-red-500"
                : "border-gray-300"
                } focus:outline-none focus:ring-2 focus:ring-blue-500`}
            />
            {errors.book_note_comments && (
              <p className="text-sm text-red-500 mt-1">
                {errors.book_note_comments.message}
              </p>
            )}
          </div>
        </div>
        {/* Buttons */}
        <div className="col-span-1 md:col-span-2 flex flex-col sm:flex-row justify-end gap-4">
          {!isEditMode && (
            <button
              type="button"
              onClick={handleReset}
              className="w-full sm:w-auto px-5 py-2 border border-red-600 text-red-600 rounded-md hover:bg-red-100 transition cursor-pointer rounded-md shadow-md text-sm font-medium transition-transform transform hover:scale-105 focus:outline-none focus:ring-2 focus:ring-red-500 focus:ring-offset-2"
            >
              Reset
            </button>
          )}
          <button
            type="submit"
            className="w-full sm:w-auto px-5 py-2  text-white rounded-md bg-gradient-to-r from-[#8C3FE1] to-[#E557AE] hover:from-[#6B3DC6] hover:to-[#C64A9B] transition shadow-sm cursor-pointer flex items-center gap-2 rounded-md shadow-md text-sm font-medium transition-transform transform hover:scale-105 focus:outline-none focus:ring-2 focus:ring-purple-500 focus:ring-offset-2 d-flex justify-center"
          >
            <span>
              {loading
                ? isEditMode
                  ? "Updating..."
                  : "Creating..."
                : isEditMode
                  ? "Update Book"
                  : "Create Book"}
            </span>
            {loading ? (
              <FiLoader size={20} className="animate-spin" />
            ) : (
              <FiArrowRightCircle size={20} />
            )}
          </button>
        </div>
      </form>
    </div>
  );
}
