"use client";
import { signIn } from 'next-auth/react';
import { useRouter } from 'next/navigation'
import React, { useState } from 'react';
import { toast } from 'react-toastify';
import Image from "next/image"
import { useForm } from 'react-hook-form';
import { zodResolver } from '@hookform/resolvers/zod';
import { loginSchema } from '@/validation/validation';
import { motion } from 'framer-motion';
import { FiMail, FiLock, FiLogIn, FiEye, FiEyeOff } from 'react-icons/fi';
// Softer, desaturated colors for background elements
const bookCoverColor = 'rgba(100, 120, 150, 0.04)'; // Very light, desaturated blue-grey
const accentColor = 'rgba(255, 120, 0, 0.03)';     // Very light, desaturated orange accent
const detailColor = 'rgba(255, 255, 255, 0.02)';   // Even lighter for text details
import { getSession } from "next-auth/react";

const FiLoader = () => (
  <svg
    xmlns="http://www.w3.org/2000/svg"
    width="20"
    height="20"
    viewBox="0 0 24 24"
    fill="none"
    stroke="currentColor"
    strokeWidth="2"
    strokeLinecap="round"
    strokeLinejoin="round"
    className="animate-spin"
  >
    <path d="M21 12a9 9 0 1 1-6.219-8.56"></path>
  </svg>
);

export default function Login() {
  const [loading, setLoading] = useState(false);
  const [showPassword, setShowPassword] = useState(false);
  const router = useRouter()

  const {
    register,
    handleSubmit,
    formState: { errors },
  } = useForm({
    resolver: zodResolver(loginSchema),
    mode: 'onBlur',
  });

  const onSubmit = async (data) => {
    setLoading(true);
    try {
      const res = await signIn("credentials", {
        redirect: false,
        email: data.email,
        password: data.password,
      });

      if (res.ok) {
        // toast.success('Login successful!');
         const session = await getSession();
         const user_role = session?.user?.role;
        console.log("User role &&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&:", session?.user?.role);
       if(user_role === 4)
       {
          router.replace('/copyediting/dashboard')
       } else {
           router.replace('/dashboard')
       }
        // window.location.href = "/dashboard";
        
        toast.success("Login successful!");
      } else {
        toast.error(res.error || 'Login failed');
      }
    } catch (error) {
      console.error('Error submitting form:', error);
      toast.error('Something went wrong. Please try again.');
    } finally {
      setLoading(false);
    }
  };

  return (
    <div className="flex h-screen w-full items-stretch justify-center font-sans relative" >
      {/* Left Side: Gradient Background with Abstract Shapes */}
      <div
        className="hidden md:flex flex-col relative w-full lg:w-1/2 p-12 bg-gradient-to-br from-[#7128DF] to-[#E557AE] text-white overflow-hidden justify-between
    before:content-[''] before:absolute before:inset-0 before:bg-black/40 before:z-10"
      > <div className="absolute inset-0 z-0" style={{
        filter: 'blur(10px)', boxShadow: `
      inset 0 0 20px rgba(0, 0, 0, 0.15),
      inset 0 0 40px rgba(0, 0, 0, 0.10),
      inset 0 0 60px rgba(0, 0, 0, 0.05)
    `,
      }}>
          <svg className="w-full h-full" viewBox="0 0 100 100" preserveAspectRatio="none" >
            <defs>
              <linearGradient id="gradient-circle-1" x1="0%" y1="0%" x2="100%" y2="100%">
                <stop offset="0%" style={{ stopColor: '#904EF8', stopOpacity: 1 }} />
                <stop offset="100%" style={{ stopColor: '#6B3DC6', stopOpacity: 1 }} />
              </linearGradient>
              <linearGradient id="gradient-circle-2" x1="0%" y1="0%" x2="100%" y2="100%">
                <stop offset="0%" style={{ stopColor: '#75B6E4', stopOpacity: 1 }} />
                <stop offset="100%" style={{ stopColor: '#4A69A5', stopOpacity: 1 }} />
              </linearGradient>
            </defs>
            <path d="M0,0 Q15,40 25,20 T40,50 T65,20 T80,45 T100,20 V0 Z" fill="#6A47DD" transform="rotate(10 50 50)" />
            <path d="M0,100 Q15,60 25,80 T40,50 T65,80 T80,55 T100,80 V100 Z" fill="#f385a8ff" transform="rotate(-10 50 50)" />
          </svg>
        </div>

        <div className="relative z-10 flex-grow flex flex-col items-center justify-center text-center px-6">
          <Image src="/logo.png" alt="logo" width={350} height={40} className="mb-6" />
          <p
            className="mt-6 text-white text-lg font-semibold"
            style={{
              fontFamily: "'Futura', sans-serif",
              textShadow: '1px 1px 4px rgba(0,0,0,0.5)',
              letterSpacing: '0.05em',
            }}
          >
            Welcome back! Access your account here.
          </p>
        </div>


        <div className="relative z-10 text-center text-sm">

          <p className="font-semibold">American Chemical Society (ACS)</p>
        </div>
      </div>

      {/* Right Side: Login Form */}
      <div className="relative flex flex-col items-center justify-center w-full lg:w-1/2 p-4 sm:p-8 md:p-12 overflow-hidden">
        {/* Animated SVG Background */}
        <div className="absolute inset-0 z-0 overflow-hidden">
          <svg className="w-full h-full" preserveAspectRatio="xMidYMid slice">
            <defs>
              {/*
                *** MODIFICATION HERE ***
                Increased width and height from 180 to 600.
                This makes the tile area significantly larger,
                so the pattern repeats much less often.
              */}
              <pattern id="neet-subtle-book-pattern" patternUnits="userSpaceOnUse" width="350" height="500" patternTransform="rotate(50)">
                <g fill="none" stroke="none"> {/* No default fill or stroke for the group */}

                  {/* All content inside the pattern remains the same, but the overall tile is now 600x600. */}

                  {/* Small NEET Closed Book 1 */}
                  <g transform="translate(20, 20) scale(0.2)"> {/* Significantly smaller scale */}
                    <rect x="0" y="0" width="100" height="140" rx="3" ry="3" fill={bookCoverColor} />
                    <rect x="10" y="15" width="80" height="20" fill={accentColor} />
                    <rect x="25" y="20" width="50" height="5" fill={detailColor} /> {/* Placeholder for "NEET" text */}
                    <rect x="15" y="45" width="70" height="3" fill={detailColor} /> {/* Subtitle line 1 */}

                    <animateTransform
                      attributeName="transform"
                      type="translate"
                      values="20 20; 22 18; 20 20"
                      dur="10s"
                      repeatCount="indefinite"
                    />
                  </g>

                  {/* Small NEET Closed Book 2 (rotated) */}
                  <g transform="translate(90, 80) scale(0.2) rotate(-15)"> {/* Slightly offset and rotated */}
                    <rect x="0" y="0" width="100" height="140" rx="3" ry="3" fill={bookCoverColor} />
                    <rect x="10" y="15" width="80" height="20" fill={accentColor} />
                    <rect x="25" y="20" width="50" height="5" fill={detailColor} /> {/* Placeholder for "NEET" text */}
                    <rect x="15" y="45" width="70" height="3" fill={detailColor} /> {/* Subtitle line 1 */}

                    <animateTransform
                      attributeName="transform"
                      type="translate"
                      values="90 80; 88 82; 90 80"
                      dur="12s"
                      repeatCount="indefinite"
                    />
                  </g>

                  {/* Small subtle atom/molecule icon */}
                  <g transform="translate(140, 30) scale(0.1)" opacity="0.4">
                    <circle cx="50" cy="50" r="40" stroke={bookCoverColor} strokeWidth="15" fill="none" />
                    <line x1="10" y1="50" x2="90" y2="50" stroke={bookCoverColor} strokeWidth="5" />
                    <line x1="50" y1="10" x2="50" y2="90" stroke={bookCoverColor} strokeWidth="5" />
                  </g>

                  {/* Small subtle microscope icon */}
                  <g transform="translate(40, 130) scale(0.1)" opacity="0.3">
                    <rect x="30" y="0" width="20" height="80" fill={bookCoverColor} rx="5" ry="5" />
                    <circle cx="40" cy="90" r="15" fill={bookCoverColor} />
                    <rect x="20" y="60" width="60" height="10" fill={bookCoverColor} rx="3" ry="3" />
                  </g>


                </g>
              </pattern>
            </defs>
            <rect width="100%" height="100%" fill="url(#neet-subtle-book-pattern)" />
          </svg>
        </div>
        <motion.div
          className="w-full max-w-md   p-8 md:p-12 relative z-10"
          animate={errors.email || errors.password ? { x: [-5, 5, -5, 5, 0] } : {}}
          transition={{ duration: 0.3, ease: "easeInOut" }}
        >
          <div
            className="mb-12 text-center"
          >
            <div className="inline-block p-4 mb-4 bg-gradient-to-r from-[#7128DF] to-[#E557AE] rounded-full">
              <FiLock className="text-white h-8 w-8" />
            </div>
            <h1 className="text-3xl font-bold text-gray-800 tracking-tight">
              Welcome Back
            </h1>
            <p className="text-md text-gray-500 mt-3">Enter your credentials to access your account.</p>
          </div>

          <form className={`space-y-6 ${loading ? 'pointer-events-none opacity-50' : ''}`} onSubmit={handleSubmit(onSubmit)}>
            {/* Email Input */}
            <div>
              <label
                htmlFor="email"
                className="block text-sm font-medium text-gray-700 mb-1"
              >
                Email Address
              </label>
              <div className="relative">
                <FiMail className="absolute top-1/2 left-3 -translate-y-1/2 text-gray-400" />
                <input
                  type="email"
                  id="email"
                  {...register('email')}
                  className={`w-full pl-10 pr-4 py-2.5 bg-slate-100/80 border-2 text-gray-800 rounded-md
                    ${errors.email ? 'border-red-400' : 'border-gray-300'} 
                    focus:outline-none focus:ring-2 focus:ring-violet-500/50 focus:border-violet-500 transition-colors`}
                  placeholder="you@example.com"
                />
              </div>
              {errors.email && <p className="text-red-500 text-xs mt-2 ml-1">{errors.email.message}</p>}
            </div>

            {/* Password Input */}
            <div>
              <label
                htmlFor="password"
                className="block text-sm font-medium text-gray-700 mb-1"
              >
                Password
              </label>
              <div className="relative">
                <FiLock className="absolute top-1/2 left-3 -translate-y-1/2 text-gray-400" />
                <input
                  type={showPassword ? "text" : "password"}
                  id="password"
                  {...register('password')}
                  className={`w-full pl-10 pr-10 py-2.5 bg-slate-100/80 border-2 text-gray-800 rounded-md
                    ${errors.password ? 'border-red-400' : 'border-gray-300'} 
                    focus:outline-none focus:ring-2 focus:ring-violet-500/50 focus:border-violet-500 transition-colors`}
                  placeholder="••••••••"
                />
                <button
                  type="button"
                  onClick={() => setShowPassword(!showPassword)}
                  className="absolute top-1/2 right-3 -translate-y-1/2 text-gray-500 hover:text-violet-600 transition-colors focus:outline-none"
                  aria-label={showPassword ? "Hide password" : "Show password"}
                >
                  {showPassword ? <FiEyeOff /> : <FiEye />}
                </button>
              </div>
              {errors.password && <p className="text-red-500 text-xs mt-2 ml-1">{errors.password.message}</p>}
            </div>
            <div className="pt-4">
              <button
                type="submit"
                disabled={loading}
                className="w-full py-3 px-4 bg-gradient-to-r from-[#7128DF] to-[#E557AE] text-white font-bold rounded-lg shadow-lg hover:shadow-violet-500/40 transition-all duration-300 transform hover:scale-105 flex items-center justify-center gap-2 focus:outline-none focus:ring-2 focus:ring-violet-500 focus:ring-offset-2 focus:ring-offset-white disabled:opacity-50 disabled:cursor-not-allowed cursor-pointer"
              >
                {loading ? (
                  <>
                    <FiLoader />
                    <span>Signing In...</span>
                  </>
                ) : (
                  <>
                    <span className="tracking-wider">LOGIN</span>
                    <FiLogIn />
                  </>
                )}
              </button>
            </div>
          </form>
        </motion.div>
      </div>
    </div >
  );
}
